-- by Northern_Strike
-- 22.07.2019

TrailerJointBlock = {}
TrailerJointBlock.modName = g_currentModName

function TrailerJointBlock.initSpecialization()
    local schemaSavegame = Vehicle.xmlSchemaSavegame
    schemaSavegame:register(XMLValueType.BOOL, "vehicles.vehicle(?).trailerJointBlock#isBlocked", "Blocked Trailer Joint.", false)
end

function TrailerJointBlock.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Trailer, specializations)
end

function TrailerJointBlock.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", TrailerJointBlock)
    SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", TrailerJointBlock)
    SpecializationUtil.registerEventListener(vehicleType, "saveToXMLFile", TrailerJointBlock)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", TrailerJointBlock)
    SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", TrailerJointBlock)
    SpecializationUtil.registerEventListener(vehicleType, "onReadStream", TrailerJointBlock)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", TrailerJointBlock)
end

function TrailerJointBlock.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "toggleTrailerBlockState", TrailerJointBlock.toggleTrailerBlockState)
end

function TrailerJointBlock:onLoad(savegame)
    local specTrailerJointBlock = self.spec_trailerJointBlock

    specTrailerJointBlock.isValid = false

    specTrailerJointBlock.isBlocked = false
    specTrailerJointBlock.lastBlockedState = false

    -- Store these here to save on performance when update loop is running
    specTrailerJointBlock.unblockText = g_i18n:getText("action_TRAILER_UNBLOCK_FRONT_JOINT", TrailerJointBlock.modName)
    specTrailerJointBlock.blockText = g_i18n:getText("action_TRAILER_BLOCK_FRONT_JOINT", TrailerJointBlock.modName)
    specTrailerJointBlock.warningText = g_i18n:getText("warning_INVALID_TRAILER_POSITION", TrailerJointBlock.modName)

    if #specTrailerJointBlock.componentJoints < 2 then
        return
    end

    if self.isServer then
        -- rotLimit is server only so need to sync 'specTrailerJointBlock.isValid'
        local rotLimit = specTrailerJointBlock.componentJoints[1].rotLimit

        if rotLimit == nil or math.deg(rotLimit[2]) == 0 then
            return --## fix for trailers without an rotary trolley
        end

        specTrailerJointBlock.rotLimitBackup = {
            rotLimit[1] or 0,
            rotLimit[2] or 0,
            rotLimit[3] or 0
        }

        specTrailerJointBlock.isValid = true
    end
end

function TrailerJointBlock:onPostLoad(savegame)
    local specTrailerJointBlock = self.spec_trailerJointBlock

    if self.isServer and specTrailerJointBlock.isValid then
        if savegame ~= nil and not savegame.resetVehicles then
            specTrailerJointBlock.isBlocked = savegame.xmlFile:getValue(savegame.key .. ".trailerJointBlock#isBlocked", specTrailerJointBlock.isBlocked)
        end
    end
end

function TrailerJointBlock:saveToXMLFile(xmlFile, key, usedModNames)
    local specTrailerJointBlock = self.spec_trailerJointBlock

    if specTrailerJointBlock.isValid then
        xmlFile:setValue(key .. "#isBlocked", specTrailerJointBlock.isBlocked)
    end
end

function TrailerJointBlock:onWriteStream(streamId, connection)
    if not connection:getIsServer() then
        local specTrailerJointBlock = self.spec_trailerJointBlock

        -- Need to sync 'isValid' here due to rotLimit only be server side
        if streamWriteBool(streamId, specTrailerJointBlock.isValid) then
            streamWriteBool(streamId, specTrailerJointBlock.isBlocked)
        end
    end
end

function TrailerJointBlock:onReadStream(streamId, connection)
    if connection:getIsServer() then
        local specTrailerJointBlock = self.spec_trailerJointBlock

        specTrailerJointBlock.isValid = streamReadBool(streamId)

        if specTrailerJointBlock.isValid then
            specTrailerJointBlock.isBlocked = streamReadBool(streamId)
        end
    end
end

function TrailerJointBlock:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    local specTrailerJointBlock = self.spec_trailerJointBlock

    if specTrailerJointBlock.isValid then
        if self.isClient then
            TrailerJointBlock.updateActionEvents(self)
        end

        if self.isServer then
            if specTrailerJointBlock.lastBlockedState ~= specTrailerJointBlock.isBlocked then
                if specTrailerJointBlock.isBlocked then
                    self:setComponentJointRotLimit(self.componentJoints[1], 2, 0, 0)
                else
                    self:setComponentJointRotLimit(self.componentJoints[1], 2, -specTrailerJointBlock.rotLimitBackup[2], specTrailerJointBlock.rotLimitBackup[2])
                end

                specTrailerJointBlock.lastBlockedState = specTrailerJointBlock.isBlocked
            end
        end
    end
end

function TrailerJointBlock:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
    local specTrailerJointBlock = self.spec_trailerJointBlock

    if self.isClient and specTrailerJointBlock.isValid then
        self:clearActionEventsTable(specTrailerJointBlock.actionEvents)

        if isActiveForInputIgnoreSelection then
            local _, actionEventId = self:addActionEvent(specTrailerJointBlock.actionEvents, InputAction.TRAILER_BLOCK_FRONT_JOINT, self, TrailerJointBlock.actionEventBlock, false, true, false, true, nil)

            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_NORMAL)
            TrailerJointBlock.updateActionEvents(self)
        end
    end
end

function TrailerJointBlock:updateActionEvents()
    local specTrailerJointBlock = self.spec_trailerJointBlock
    local actionEvent = specTrailerJointBlock.actionEvents[InputAction.TRAILER_BLOCK_FRONT_JOINT]

    if actionEvent ~= nil then
        if self.isActive then
            g_inputBinding:setActionEventText(actionEvent.actionEventId, specTrailerJointBlock.isBlocked and specTrailerJointBlock.unblockText or specTrailerJointBlock.blockText)
        end

        g_inputBinding:setActionEventActive(actionEvent.actionEventId, self.isActive)
    end
end

function TrailerJointBlock:toggleTrailerBlockState(isBlocked, noEventSend)
    local specTrailerJointBlock = self.spec_trailerJointBlock

    if isBlocked ~= specTrailerJointBlock.isBlocked then
        specTrailerJointBlock.isBlocked = isBlocked

        if noEventSend == nil or noEventSend == false then
            if g_server ~= nil then
                g_server:broadcastEvent(TrailerJointBlockEvent.new(self, isBlocked), nil, nil, self)
            else
                g_client:getServerConnection():sendEvent(TrailerJointBlockEvent.new(self, isBlocked))
            end
        end
    end
end

function TrailerJointBlock.actionEventBlock(self, actionName, inputValue, callbackState, isAnalog)
    local specTrailerJointBlock = self.spec_trailerJointBlock

    local _, y1, _ = getRotation(self.components[1].node)
    local _, y2, _ = getRotation(self.components[2].node)

    if math.abs(y1 - y2) < 0.07 then
        self:toggleTrailerBlockState(not specTrailerJointBlock.isBlocked)
    else
        g_currentMission:showBlinkingWarning(specTrailerJointBlock.warningText, 2000)
    end
end

--------------
-- MP event --
--------------

TrailerJointBlockEvent = {}
local TrailerJointBlockEvent_mt = Class(TrailerJointBlockEvent, Event)

InitEventClass(TrailerJointBlockEvent, "TrailerJointBlockEvent")

function TrailerJointBlockEvent.emptyNew()
    local self = Event.new(TrailerJointBlockEvent_mt)

    return self
end

function TrailerJointBlockEvent.new(trailer, isBlocked)
    local self = TrailerJointBlockEvent.emptyNew()

    self.trailer = trailer
    self.trailerIsBlocked = isBlocked

    return self
end

function TrailerJointBlockEvent:readStream(streamId, connection)
    self.trailer = NetworkUtil.readNodeObject(streamId)
    self.trailerIsBlocked = streamReadBool(streamId)

    self:run(connection)
end

function TrailerJointBlockEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.trailer)
    streamWriteBool(streamId, self.trailerIsBlocked)
end

function TrailerJointBlockEvent:run(connection)
    if not connection:getIsServer() then
        g_server:broadcastEvent(TrailerJointBlockEvent.new(self.trailer, self.trailerIsBlocked), nil, connection, self.trailer)
    end

    if self.trailer ~= nil then
        self.trailer:toggleTrailerBlockState(self.trailerIsBlocked, true)
    end
end
