
-- ============================================================================
-- === MinIdleEngineLoad.lua
-- === Mod by [LSMT] Modding Team 
-- === LS25 /FS25
-- === Script by [LSMT] BaTt3RiE @ 2025
-- === Ver 1.0.0.0
-- === Erzwingt min. 4-5% Motorlast bei laufendem Motor wenn Motorlast 0.00 (globaler Patch mit Schwankung)
-- ============================================================================

if g_MinIdleEngineLoad ~= nil then
    -- Doppelladen verhindern (falls mehrere Mods das Script referenzieren)
    return
end
g_MinIdleEngineLoad = true

local MIN_IDLE_LOAD = 0.04              -- % Mindestlast
local MAX_IDLE_LOAD = 0.05              -- % Maximallast
            
local IDLE_VARIATION_SPEED = 0.0010     -- Geschwindigkeit der Schwankung
local UPDATEFREQUENCY = 10              -- Update-Intervall in ms original: 50ms

-- Speicher für Schwankungswerte pro Fahrzeug
local idleLoadVariations = {}

local function isMotorOn(self)
    if self.getMotorState == nil then return false end
    local st = self:getMotorState()
    return st == MotorState.ON
end

-- Berechne schwankende Leerlauflast für Fahrzeug
local function getIdleLoadForVehicle(vehicle, dt)
    local vehicleId = tostring(vehicle)
    
    -- Initialisiere wenn noch nicht vorhanden
    if idleLoadVariations[vehicleId] == nil then
        idleLoadVariations[vehicleId] = {
            currentLoad = MIN_IDLE_LOAD,
            direction = 1,  -- 1 = aufwärts, -1 = abwärts
            lastUpdateTime = 0
        }
    end
    
    local data = idleLoadVariations[vehicleId]
    
    -- Update alle 50ms für flüssige Schwankung
    local currentTime = g_currentMission.time
    if currentTime - data.lastUpdateTime > UPDATEFREQUENCY then
        -- Schwanke in aktuelle Richtung
        data.currentLoad = data.currentLoad + (IDLE_VARIATION_SPEED * data.direction)
        
        -- Richtungswechsel an Grenzen
        if data.currentLoad >= MAX_IDLE_LOAD then
            data.currentLoad = MAX_IDLE_LOAD
            data.direction = -1
        elseif data.currentLoad <= MIN_IDLE_LOAD then
            data.currentLoad = MIN_IDLE_LOAD
            data.direction = 1
        end
        
        data.lastUpdateTime = currentTime
    end
    
    return data.currentLoad
end

-- 1) Anzeigen/Audio/API: getMotorLoadPercentage() -> schwankende Last
Motorized.getMotorLoadPercentage = Utils.overwrittenFunction(
    Motorized.getMotorLoadPercentage,
    function(self, superFunc)
        local load = superFunc(self)
        
        if isMotorOn(self) and (load == nil or load < MIN_IDLE_LOAD) then
            return getIdleLoadForVehicle(self)
        end
        
        return load
    end
)

-- 2) Interne Werte nach dem Motorized-Update mit Schwankung setzen
Motorized.onUpdate = Utils.overwrittenFunction(
    Motorized.onUpdate,
    function(self, superFunc, dt, isActiveForInput, isSelected, isOnActiveVehicle)
        -- Erst die originale Logik laufen lassen
        superFunc(self, dt, isActiveForInput, isSelected, isOnActiveVehicle)

        local spec = self.spec_motorized
        if spec == nil or not isMotorOn(self) then
            return
        end

        -- Hole schwankende Leerlauflast für dieses Fahrzeug
        local idleLoad = getIdleLoadForVehicle(self, dt)

        -- Clientseitig: geglättete Last für HUD/Sounds
        if spec.smoothedLoadPercentage ~= nil and spec.smoothedLoadPercentage < MIN_IDLE_LOAD then
            spec.smoothedLoadPercentage = idleLoad
        end

        -- Serverseitig: tatsächliche Last + Rohwert
        if self.isServer then
            if spec.actualLoadPercentage ~= nil and spec.actualLoadPercentage < MIN_IDLE_LOAD then
                spec.actualLoadPercentage = idleLoad
            end
            if spec.motor ~= nil and spec.motor.rawLoadPercentage ~= nil and spec.motor.rawLoadPercentage < MIN_IDLE_LOAD then
                spec.motor.rawLoadPercentage = idleLoad
            end
        end
    end
)

-- Cleanup wenn Fahrzeug gelöscht wird (optional, verhindert Memory Leak)
local originalOnDelete = Vehicle.onDelete
Vehicle.onDelete = function(self)
    local vehicleId = tostring(self)
    if idleLoadVariations[vehicleId] ~= nil then
        idleLoadVariations[vehicleId] = nil
    end
    
    if originalOnDelete then
        originalOnDelete(self)
    end
end